#!/usr/bin/perl -w 

package esmith::FormMagick::Panel::userpanelViewgroups;

use strict;

use esmith::FormMagick;
use esmith::ConfigDB;
use esmith::AccountsDB;
use File::Basename;
use Exporter;
use Carp;
    
our @ISA = qw(esmith::FormMagick Exporter);

our @EXPORT = qw(
  show_initial
  genUsers
  get_accounts_prop
  get_description
  get_group_mail
);  
    
our $accounts = esmith::AccountsDB->open() || die "Couldn't open accounts";
our $db = esmith::ConfigDB->open || die "Couldn't open config db";

our $VERSION = sprintf '%d.%03d', q$Revision: 1.38 $ =~ /: (\d+).(\d+)/;


sub new {
    shift;
    my $self = esmith::FormMagick->new();
    $self->{calling_package} = (caller)[0];
    bless $self;
    return $self;
}


=head2 get_accounts_prop ITEM PROP
    
A simple accessor for esmith::AccountsDB::Record::prop

=cut

sub get_accounts_prop {
    my $fm   = shift;
    my $item = shift;
    my $prop = shift;

    my $record = $accounts->get($item);

    if ($record) {
        return $record->prop($prop);
    }   
    else {
        return '';
    }
}

=head2 get_description

Get the Description for the group named in the CGI argument "GroupName"

=cut

sub get_description {
    my $fm    = shift;
    my $group = $fm->{'cgi'}->param('groupName');
    return ( $fm->get_accounts_prop( $group, 'Description' ) );
}

=head2 get_group_mail

Get the mail address for the group named in the CGI argument "groupName"

=cut

sub get_group_mail {
    my $fm    = shift;
    my $group = $fm->{'cgi'}->param('groupName');
    # Get all the pseudonymes
    my $domain = $db->get('DomainName')->value();
    my @mails = ("$group\@$domain");
    foreach ($accounts->pseudonyms()){
        next unless (($_->prop('Account') || '') eq $group);
        my $mail = $_->key;
        $mail .= "\@$domain" if ($mail !~ /\@/);
        push @mails, $mail;
    }
    return join("<br>",@mails);
}

=head1 ACTION


=head2 show_initial FM

Show the "start" page for this panel

=cut

sub show_initial () {
    my $fm = shift;
    my $q = $fm->{cgi};
    $q->Delete('groupName');

    my $params = $fm->build_cgi_params();

    my $numGroups = $accounts->groups;

    if ( $numGroups == 0 ) {
        print $q->Tr($q->td(
            '<p><b>' . $fm->localise("ACCOUNT_GROUP_NONE") . '</p></b>'));

    }
    else {
        print $q->Tr($q->td({-colspan => 2}, $fm->localise('CURRENT_LIST')));
        print $q->start_table({-CLASS => "sme-border"}),"\n";
        print "<tr><th class=\"sme-border\">"
            . $fm->localise("GROUP")
            . "</th> <th class=\"sme-border\">"
            . $fm->localise('DESCRIPTION')
            . "</th><th class=\"sme-border\" colspan=\"2\">"
            . $fm->localise('ACTION')
            . "</th></tr>";
        foreach my $group ( $accounts->groups() ) {
            $params = $fm->build_cgi_params( $group->key );
            print "<tr>" . "<td class=\"sme-border\">"
              . $group->key . "</td>" . "<td class=\"sme-border\">"
              . $group->prop('Description') . "</td>"
              . "<td class=\"sme-border\"><a href=\"userpanel-viewgroups?$params&wherenext=ViewGroup\">"
              . $fm->localise("VIEW") . "</a></td>"

        }
        print $q->end_table,"\n";
    }
    return;
}

=head2 build_cgi_params()

Builds a CGI query string, using various sensible
defaults and esmith::FormMagick's props_to_query_string() method.

=cut

sub build_cgi_params {
    my ( $fm, $group ) = @_;

    my %props = (
        page       => 0,
        page_stack => "",
        ".id"      => $fm->{cgi}->param('.id') || "",
        groupName  => $group,
    );

    return $fm->props_to_query_string( \%props );
}

=head2 genUsers MEMBERS

Takes a comma delimited list of users and returns a string of 
html checkboxes for all system users with the members of the group
in $fm->{cgi}->parm('groupName')checked.

=cut

sub genUsers () {
    my $fm = shift;
    my $members = "";
    my $group = $fm->{'cgi'}->param('groupName');

    if ($accounts->get($group)) {
      $members = $accounts->get($group)->prop('Members');
    }
    my @members = split(/[,;]/, $members);

    my $out = "<tr>\n        <td class=\"sme-noborders-label\">"
      . $fm->localise('GROUP_MEMBERS')
      . "</td>\n        <td>\n"
      . "          <table border='0' cellspacing='0' cellpadding='0'>\n";
    foreach my $user (@members) {
        my $name = $accounts->get($user)->prop('FirstName') . " " . $accounts->get($user)->prop('LastName');

        $out .="            <tr>\n"
          . "<td>$name (".$user.")</td>\n            </tr>\n";

    }

    $out .= "          </table>\n        </td>\n    </tr>\n";
    return $out;
}

1;
